
/* Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_tlist.c */

#include "fs_itype.h"

#if defined(FS_BITMAPS)

#include "fs_bitmap.h"

static TLIST *size_tlist(_DS_ TLIST *list, FS_SHORT max, FS_SHORT num)
{
    FS_SHORT i;
    FS_LONG n;

    list->max = max;
    list->num = num;
    list->ref_count++; /* prevent get_some_back from killing tlists */

    list->indices = (FS_LONG *)(list->indices);
    if (list->numIndices < num)
    {
        list->numIndices = num;

        /* need <num> starting indices */

        /* realloc, except no need for data copy, so free and malloc */
        FSS_free(_PS_ list->indices);

#ifdef FS_MEM_DBG
        STATE.memdbgid = "TLIST(indices)";
#endif
        list->indices = (FS_LONG *) FSS_malloc(_PS_ num * sizeof(FS_LONG));
        if (list->indices == 0)
        {
            list->ref_count--;
            delete_tlist(_PS_ list);
            return 0;
        }
    }
    /* all initially UNUSED */
    for (i = 0; i < num; i++)
        list->indices[i] = -1;

    /* start off with eight transitions per row ... might be enough */
    n = (FS_LONG)num << 3;
    list->stop_index = n;
    list->next_index = 0;

    list->nodes = (TNODE *)(list->nodes);
    if (list->numNodes < n)
    {
        list->numNodes = n;

        /* realloc, except no need for data copy, so free and malloc */
        FSS_free(_PS_ list->nodes);

#ifdef FS_MEM_DBG
        STATE.memdbgid = "TLIST(nodes)";
#endif
        list->nodes = (TNODE *) FSS_malloc(_PS_ n * sizeof(TNODE));
        if (list->nodes == 0)
        {
            list->ref_count--;
            delete_tlist(_PS_ list);
            return 0;
        }
    }
    list->ref_count--;

    return list;
}

/****************************************************************/
/* Each row of the forthcoming bitmap has it's own singly linked
* list of transition x coordinates. These coordinates are
* maintained in ascending order. The low order bit is the
* direction UP/DN of the segment which generated the transition,
* necessary for computation of non-zero-winding number.
*/
/****************************************************************/
/* allocate a new list */
TLIST *new_tlist(_DS_ TLIST *olist, FS_SHORT max, FS_SHORT num)
{
    FS_SHORT i;
    FS_LONG n;
    TLIST *list;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "TLIST";
#endif
    if (olist)
    {
        return size_tlist(_PS_ olist, max, num);
    }
    list = (TLIST *) FSS_calloc(_PS_ sizeof(TLIST));

    if (list == 0)
        return 0;


    list->max = max;
    list->num = num;
    list->ref_count = 0;

    list->numIndices = num;

    /* need <num> starting indices */

#ifdef FS_MEM_DBG
    STATE.memdbgid = "TLIST(indices)";
#endif
    list->indices = (FS_LONG *) FSS_malloc(_PS_ num * sizeof(FS_LONG));
    if (list->indices == 0)
    {
        delete_tlist(_PS_ list);
        return 0;
    }

    /* all initially UNUSED */
    for (i = 0; i < num; i++)
        list->indices[i] = -1;

    /* start off with eight transitions per row ... might be enough */
    n = (FS_LONG)num << 3;
    list->stop_index = n;
    list->next_index = 0;

    list->numNodes = n;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "TLIST(nodes)";
#endif
    list->nodes = (TNODE *) FSS_malloc(_PS_ n * sizeof(TNODE));

    if (list->nodes == 0)
    {
        delete_tlist(_PS_ list);
        return 0;
    }
    return list;
}

/****************************************************************/
/* FSS_free a list */
FS_VOID delete_tlist(_DS_ TLIST *list)
{
    if (list)
    {
        FSS_free(_PS_ (FS_LONG *)(list->indices));
        list->indices = 0;
        FSS_free(_PS_ (TNODE *)(list->nodes));
        list->nodes = 0;
        FSS_free(_PS_ list);
    }
}

/****************************************************************/
/* insert an element into a sublist maintaining sorted order */
/* (x >> 1) == coordinate, (x & 1) == UP */
FS_VOID append_tlist(_DS_ FS_FIXED x, FS_SHORT y, TLIST *list)
{
    FS_SHORT row;
    FS_LONG index, next;
    TNODE *nodes = (TNODE *)(list->nodes);

    /* ? nodes full */
    if (list->next_index == list->stop_index)
    {
        FS_LONG stop_index;
        stop_index = list->stop_index + list->num;

        if (list->numNodes < stop_index)
        {
#ifdef FS_MEM_DBG
            STATE.memdbgid = "TLIST(nodes)";
#endif
            nodes = (TNODE *)FSS_realloc(_PS_ (FS_VOID *)nodes, stop_index * sizeof(TNODE));
            if (nodes == 0)
            {
                return;
            }
            list->numNodes = stop_index;
        }
        /* now update the lists nodes + stop_index */
        list->stop_index = stop_index;
        list->nodes = nodes;
    }

    /*  this row's list starts at ... */
    row = list->max - y;
    if (row >= list->num || row < 0)
    {
        /* ??? maybe this should be a fatal error ??? */
#ifdef FS_DEBUG
        FS_PRINTF(("*** append_tlist: bad y coordinate %d\n", y));
#endif
        return;
    }
    index = list->indices[row];

    /* add the element to the list */
    if (index < 0 || nodes[index].value > x)
    {
        /* prepend new node to list, store new head */
        nodes[list->next_index].value = x;
        nodes[list->next_index].next = index;
        list->indices[row] = list->next_index++;
    }
    else
    {
        /* move to the proper <index> */
        while ((next = nodes[index].next) >= 0 && nodes[next].value <= x)
            index = next;

        /* insert new node after <index> */
        nodes[index].next = list->next_index;
        nodes[list->next_index].value = x;
        nodes[list->next_index].next = next;
        list->next_index++;
    }
}

/****************************************************************/

FS_VOID remove_tran(FS_FIXED x, FS_SHORT y, TLIST *list)
{
    FS_LONG row;
    FS_LONG index, next, prev = -1;
    TNODE *nodes = (TNODE *)(list->nodes);

    row = list->max - y;
    if (row < 0 || row >= list->num)
        return;

    index = list->indices[row];
    if (index < 0 || index > list->stop_index)
        return;
    if (nodes[index].value == x)
    {
        /* first one is one to remove */
        list->indices[row] = nodes[index].next;
    }
    else
    {
        while ((next = nodes[index].next) >= 0 && nodes[index].value != x)
        {
            prev = index;
            index = next;
        }
        if (index >= 0 && prev >= 0 && nodes[index].value == x)
            nodes[prev].next = nodes[index].next;
    }
}


/****************************************************************/
/* display a list in ASCII */
#ifdef FS_DUMP
FS_VOID dump_tlist(FILECHAR *s, TLIST *list)
{
    int i, index;

    FS_PRINTF(("dump_tlist: %s\n", s));
    FS_PRINTF(("next_index=%ld stop_index=%ld num=%d\n",
               (long)list->next_index, (long)list->stop_index, list->num));

    for (i = 0; i < list->num; i++)
    {
        FS_PRINTF(("%d:", list->max - i));
        index = list->indices[i];
        while (index >= 0)
        {
            FS_FIXED x;
            x = list->nodes[index].value;
            FS_PRINTF(("%c%f ", (x & 1) ? 'U' : 'D', (x >> 1) / 65536.0));
            index = list->nodes[index].next;
        }
        FS_PRINTF(("\n"));
    }
}
#endif /* FS_DUMP */
/****************************************************************/
#endif /* if defined(FS_BITMAPS) || defined(FS_GRAYMAPS) */
